/* $Header: /CVSROOT/gcopy/gcopy.c,v 1.5 2008-10-20 01:35:20 tino Exp $
 *
 * Generalized Copy (currently evolved from mvatom)
 *
 * Copyright (C)2008 Valentin Hilbig <webmaster@scylla-charybdis.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 *
 * $Log: gcopy.c,v $
 * Revision 1.5  2008-10-20 01:35:20  tino
 * Bugfix for two arg mode
 *
 * Revision 1.4  2008-10-20 01:09:50  tino
 * More usage fixes
 *
 * Revision 1.3  2008-10-20 01:07:51  tino
 * Usage corrected
 *
 * Revision 1.2  2008-10-20 00:56:23  tino
 * Open O_DIRECT skipped on filesystems lacking support
 *
 * Revision 1.1  2008-10-20 00:39:21  tino
 * First dist
 *
 */

#include "tino/alarm.h"
#include "tino/filetool.h"
#include "tino/getopt.h"
#include "tino/buf_line.h"
#include "tino/scale.h"

#include <sys/time.h>

#include "gcopy_version.h"

static int		errflag;
static int		g_backup, g_ignore, g_nulls, g_lines, g_quiet, g_verbose, g_mkdirs, g_noprogress;
static int		g_isroot;
static size_t		g_memory;
static const char	*g_dest, *g_source, *g_backupdir;
static void		*g_block;


/**********************************************************************/

static void
verror_fn(const char *prefix, TINO_VA_LIST list, int err)
{
  if (!g_quiet)
    tino_verror_ext(list, err, "gcopy %s", prefix);
  if (!g_ignore)
    exit(1);
  errflag	= 1;
}

static void
verbose(const char *s, ...)
{
  tino_va_list	list;

  if (!g_verbose)
    return;

  tino_va_start(list, s);
  vprintf(s, tino_va_get(list));
  tino_va_end(list);
  printf("\n");
}

static unsigned long long	p_total;
static char			p_mode;
static int			p_count, p_all;
static tino_file_size_t		p_pos, p_max;

static int
progress_cb(void *user, long delta, time_t now, long run)
{
  static int	lcnt;
  static time_t	last;

  xDP(("(%p %ld %ld %ld)", user, delta, now, run));
  if (lcnt!=p_count)
    {
      lcnt	= p_count;
      last	= now;
    }
  fprintf(stderr
	  , "%c %s %d/%d %s%% %s/%s %s/s %s \r"
	  , p_mode
	  , tino_scale_interval(1, run, 1, -6)
	  , p_count, p_all
	  , tino_scale_percent(2, p_pos, p_max, -4)
	  , tino_scale_bytes(3, p_pos, 2, -6)
	  , tino_scale_bytes(4, p_max, 2, -6)
	  , tino_scale_speed(5, p_pos, now-last, 1, -6)
	  , tino_scale_bytes(6, p_total, 2, -6)
	  );
  fflush(stderr);
  xDP(("() 0"));
  return 0;
}

static void
set_mode(char c)
{
  p_mode	= c;
  TINO_ALARM_RUN();
}


/**********************************************************************/

static void
run_copy(int fd_src, int fd_dest, const char *src, const char *dest)
{
  verbose("copy: %s -> %s", src, dest);

  p_count++;
  p_pos	= 0;
  for (;;)
    {
      int	got;

      set_mode('R');

      if ((got=tino_file_read_allE(fd_src, g_block, g_memory))==0)
	break;

      if (got<0)
	{
	  tino_err("ETTGC100B read error in %s at %llu", src, (unsigned long long)p_pos);
	  return;
	}

      set_mode('W');

      if (tino_file_write_allE(fd_dest, g_block, got)!=got)
	{
	  tino_err("ETTGC101B write error in %s at %llu", src, (unsigned long long)p_pos);
	  return;
	}

      p_pos	+= got;
      p_total	+= got;
    }
}

static int
try_open_read(const char *name)
{
  int	fd;

  if ((fd=tino_file_openE(name, O_RDONLY|O_NOFOLLOW|O_DIRECT))<0)
    fd	= tino_file_openE(name, O_RDONLY|O_NOFOLLOW);
  return fd;
}


static void
do_copy(const char *src, const char *dest)
{
  int			fd_src, fd_dest;
  tino_file_stat_t	st;
  struct timeval	tv[2];
  int			mode;

  set_mode('O');

  if ((fd_src=try_open_read(src))<0)
    {
      tino_err("ETTGC110A cannot open source %s", src);
      return;
    }
  if (tino_file_stat_fdE(fd_src, &st))
    {
      tino_err("ETTGC111A cannot stat source %s", src);
      return;
    }
  p_max	= st.st_size;
  if ((fd_dest=tino_file_open_createE(dest, O_WRONLY|O_EXCL|O_NOFOLLOW, 0600))<0)	/* O_SYNC is slow	*/
    {
      tino_err("ETTGC112A cannot create destination %s", dest);
      tino_file_closeE(fd_dest);
      return;
    }

  set_mode('M');

  run_copy(fd_src, fd_dest, src, dest);

  set_mode('C');

  tv[0].tv_sec	= st.st_atime;
  tv[1].tv_sec	= st.st_mtime;
  tv[0].tv_usec	= 0;
  tv[1].tv_usec	= 0;
  if (futimes(fd_dest, tv))
    tino_err("ETTGC113B cannot set file times of %s", dest);

  if (fchown(fd_dest, st.st_uid, st.st_gid))
    {
      if (errno!=EPERM || g_isroot)
	tino_err("ETTGC114A cannot set ownership of %s", dest);
      else
	verbose("ITTGC114 cannot set ownership of %s", dest);

      /* do not set dangerous modes on ownership failure
       */
      mode	= st.st_mode&0777;
    }
  else
    mode	= st.st_mode&07777;

  if (fchmod(fd_dest, mode))
    tino_err("ETTGC115A cannot set file mode of %s", dest);

  if (tino_file_closeE(fd_dest))
    tino_err("ETTGC116B error closing destination %s", dest);
  if (tino_file_closeE(fd_src))
    tino_err("ETTGC117B error closing source %s", src);
}


/**********************************************************************/

static void
do_rename(const char *name, const char *to)
{
  if (rename(name, to))
    {
      tino_err("ETTGC120A cannot rename %s -> %s", name, to);
      return;
    }
  verbose("rename: %s -> %s", name, to);
}

static void
do_rename_away(const char *name, const char *rename)
{
  char	*tmp, *o;

  o	= 0;
  if (!g_backupdir || !tino_file_notexistsE(rename=tmp=o=tino_file_glue_pathOi(NULL, 0, g_backupdir, tino_file_filenameptr_constO(rename))))
    tmp	= tino_file_backupnameNi(NULL, 0, rename);
  do_rename(name, tmp);
  tino_freeO(tmp);
  if (o!=tmp)
    tino_freeO(o);
}

/**********************************************************************/

static void
do_mkdirs(const char *path, const char *file)
{
  switch (tino_file_mkdirs_forfileE(path, file))
    {
    case -1:
      tino_err("ETTGC130A failed: mkdir for %s%s%s", path ? path : "", path ? "/" : "", file);
      break;
      
    case 1:
      verbose("mkdir for %s%s%s", path ? path : "", path ? "/" : "", file);
      break;
    }
}

/* This actually is a hack.
 *
 * We only have one single operation active at a time.  So we can use
 * a static buffer here which keeps the intermediate string.
 */
static const char *
get_src(const char *name)
{
  static TINO_BUF	buf;

  if (!g_source)
    return name;

  tino_buf_resetO(&buf);
  tino_buf_add_sO(&buf, g_source);
  tino_buf_add_sO(&buf, name);
  return tino_buf_get_sN(&buf);
}

static void
do_gcopy_backup(const char *old, const char *new)
{
  const char	*src;

  src	= get_src(old);

  /* Try to figure out what happened	*/
  if (tino_file_notexistsE(src))
    {
      tino_err("ETTGC140B missing old name for rename: %s", src);
      return;
    }
  errno	= 0;	/* Do not report errors in case there is no error	*/

  if (!tino_file_notexistsE(new))
    {
      000;	/* Compare here	*/
      000;	/* Append mode here	*/

      if (!g_backup)
	{
	  tino_err("ETTGC141A existing destination: %s", new);
	  return;
	}
      do_rename_away(new, new);
    }
  else if (g_mkdirs)
    do_mkdirs(NULL, new);

  do_copy(src, new);
}


/**********************************************************************/

static const char *
read_dest(void)
{
  static TINO_BUF buf;

  if (!g_nulls && !g_lines)
    {
      tino_err("FTTGC201 missing option -l or -0 to read stdin");
      return 0;
    }
  return tino_buf_line_read(&buf, 0, (g_nulls ? 0 : '\n'));
}

static void
do_dest(const char *name)
{
  const char	*targ;
  char		*dest;
 
  targ	= tino_file_filenameptr_constO(name);
  dest	= tino_file_glue_pathOi(NULL, 0, g_dest, targ);
  do_gcopy_backup(name, dest);
  tino_freeO(dest);
}

static void
gcopy1(const char *name)
{
  if (g_mkdirs<2 && tino_file_notdirE(g_dest))
    {
      tino_err((tino_file_notexistsE(g_dest)
		? "ETTGC150A missing destination directory: %s"
		: "ETTGC151A existing destination not a directory: %s"
		), g_dest);
      return;
    }
  if (strcmp(name, "-"))
    do_dest(name);
  else
    while ((name=read_dest())!=0)
      do_dest(name);
}


/**********************************************************************/

static int
is_directory_target(const char *name)
{
  const char	*tmp;

  tmp	= tino_file_filenameptr_constO(name);
  if (*tmp && strcmp(tmp, ".") && strcmp(tmp, ".."))
    return 0;
  return !tino_file_notdirE(name);
}

int
main(int argc, char **argv)
{
  int		argn, g_original;
  unsigned long	memory;

  tino_verror_fn	= verror_fn;

  argn	= tino_getopt(argc, argv, 1, 0,
		      TINO_GETOPT_VERSION(GCOPY_VERSION)
		      " name.. [dest]\n"
		      "	if name is - lines are read from stdin.\n"
		      "	example:	gcopy -v source target\n"
		      "	convenience:	alias cp='gcopy -o'",

		      TINO_GETOPT_USAGE
		      "h	this help"
		      ,
		      TINO_GETOPT_FLAG
		      "0	(This option is 'number zero', not a big O!)\n"
		      "		read 0 terminated lines from stdin\n"
		      "		example: find . -type f -print0 | gcopy -0ab -"
		      , &g_nulls,

		      /* 1-9 undefined */

#if 0
		      TINO_GETOPT_FLAG
		      "a	Check all backups for matches"
		      , &g_compare_all,
#endif
		      TINO_GETOPT_FLAG
		      "b	Backup existing destination to .~#~\n"
		      "		On errors this might leave you with a renamed destination!\n"
		      "		Cannot be used together with option -a"
		      , &g_backup,

		      TINO_GETOPT_STRING
		      "c dir	Create backups in the given directory.\n"
		      "		This moves an existing destination into the given dir,\n"
		      "		possibly renaming it according to options -a and -b"
		      , &g_backupdir,

		      TINO_GETOPT_STRING
		      "d dir	target (Destination) directory to move files into"
		      , &g_dest,

		      /* E Error recovery (loop on errors)	*/
		      /* F Force	*/
		      /* G Gcopy-Option	*/

		      TINO_GETOPT_FLAG
		      "i	Ignore (common) errors"
		      , &g_ignore,

		      /* J future use	*/

		      TINO_GETOPT_FLAG
		      "l	read Lines from stdin, enables '-' as argument\n"
		      "		example: find . -print | gcopy -lb -"
		      , &g_lines,

		      TINO_GETOPT_ULONGINT
		      TINO_GETOPT_DEFAULT
		      TINO_GETOPT_SUFFIX
		      TINO_GETOPT_MIN
		      TINO_GETOPT_MAX
		      "m N	size of Memory block for copy operation.\n"
		      "		Must be a multiple of 4096, suffix hint: BSKCMGTPEZY"
		      , &memory,
		      1024*1024,
		      4096,
		      1024*1024*1024,

		      TINO_GETOPT_FLAG
		      "n	No progress indicator"
		      , &g_noprogress,

		      TINO_GETOPT_FLAG
		      "o	Original behavior if directory is the last target\n"
		      "		The last argument must end in a / or must be . or .."
		      , &g_original,

		      TINO_GETOPT_FLAG
		      TINO_GETOPT_MAX
		      "p	create missing Parent directories (for -r)\n"
		      "		Give twice to create for -d, too"
		      , &g_mkdirs,
		      2,

		      TINO_GETOPT_FLAG
		      "q	silently fail"
		      , &g_quiet,
#if 0
		      TINO_GETOPT_FLAG
		      "r	Recursive mode, copy full directory contents"
		      , &g_recurse,
#endif
		      TINO_GETOPT_STRING
		      "s src	append the given Src prefix, for an usage like:\n"
		      "		( cd whatever; ls -1; ) | gcopy -l -s whatever/ -d todir -\n"
		      "		The source prefix is not a directory, it's a literal prefix"
		      , &g_source,

		      /* T Timeout	*/
#if 0
		      TINO_GETOPT_FLAG
		      "u	Use umask, do not copy permissions nor capabilities"
		      , &g_umask,
#endif
		      TINO_GETOPT_FLAG
		      "v	verbose"
		      , &g_verbose,
#if 0
		      TINO_GETOPT_FLAG
		      "x	Extended mode, copy non-file types (dev, softlinks, etc.)"
		      , &g_extended,
#endif
#if 0
		      /* W web		*/
		      /* Y yes mode	*/
		      /* Z (compress)	*/
#endif
		      NULL);

  if (argn<=0)
    return 1;

  g_isroot	= !geteuid();

  if (g_original && !g_dest && argn+1<argc && is_directory_target(argv[argc-1]))
    g_dest	= argv[--argc];

  if (!g_noprogress)
    tino_alarm_set(1, progress_cb, NULL);

  g_memory	= memory;
  memory	= tino_alloc_align_size(&g_memory);
  if (memory && !g_quiet)
    fprintf(stderr, "WTTGC202 Blocksize was rounded up %ld byte to %s\n",
	    memory, tino_scale_bytes(0, g_memory, 2, 6));

  g_block	= tino_alloc_alignedO((size_t)g_memory);

  p_all		= 1;
  if (g_dest)
    {
      p_all	= argc-argn;
      while (argn<argc)
	gcopy1(argv[argn++]);
    }
  else if (argc==argn+2)
    do_gcopy_backup(argv[argn], argv[argn+1]);
  else
    tino_err("FTTGC200 Two arguments are needed when neither -d nor -o is present");

  tino_freeO(g_block);

  return errflag;
}
